#!/usr/bin/perl

package FCC::tix;

#######################################
#                                     #
#     FCC Ledger-blocks               #
#     (all big-endian)                #
#                                     #
#    (C) 2018 Domero, Chaosje         #
#                                     #
#######################################

use strict;
use warnings;
use Exporter;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

$VERSION     = '1.01';
@ISA         = qw(Exporter);
@EXPORT      = qw();
@EXPORT_OK   = qw(
  saldo_detail read_fromaddress
  read_tix
  read_prev read_next read_tid read_cumhash read_num read_version read_type read_prevtid read_fcctime
  read_outblocks
  read_coinbase_count read_coinbase_signature
  read_in_blocks read_in_signature read_in_block
  read_out_address read_out_amount read_out_fee
);

use FCC::global;
use FCC::fcc qw(walletposlist);

################################################################################

my $TIX = {
# name                => [ offset , length , content-description ]
  # ========================================================================== #
  # All Blocks
  # ========================================================================== #
  prev               => [0,4,"difference of position of the previous transaction in the chain compared to this position (equal to length previous block)"],
  next               => [4,4,"difference of position of the next transaction in the chain compared to this position (equal to length block)"],
  tid                => [8,64,"transaction id (next fields from 'number' on, secure-hashed (sha256(sha512(data)))"],
  cumhash            => [72,64,"cumulative hash of all blocks"],
  num                => [136,12,"transaction number in chain (0=genesis) 48 bit"],
  version            => [148,4,"version"],
  type               => [152,1,"Block Type"],
  prevtid            => [153,64,"previous id (any transaction type)"],
  # ========================================================================== #
  # Genesis / In / Coinbase
  # ========================================================================== #
  fcctime            => [217,8,"time, epoch is 00:00:00 UTC, January 1, 1970 (UNIX)"],
  outblocks          => [225,2,"number of out addresses ( = 1 for genesis/coinbase)"],
  # ========================================================================== #
  # Coinbase
  # ========================================================================== #
  coinbase_count     => [227,8,"Counbase Count"],
  coinbase_signature => [235,128,"signature of FCC-Server"],
  # ========================================================================== #
  # In
  # ========================================================================== #
  in_blocks          => [227,2,"number of in addresses"],
  in_signature       => [229,192,"signature (public key (64) followed by signature (128))"],
  in_block           => [421,64," * in_blocks, list of id's of collected out-transactions to form in-addresses"],
  # ========================================================================== #
  # Out
  # ========================================================================== #
  out_address        => [217,68,"FCC-wallet address"],
  out_amount         => [285,16,"amount in 64 bit (100000000 = 1 FCC)"],
  out_fee            => [301,4,"fee = 0 for coinbase and genesis, minimum = 50 = 0,5% maximum = 10000 (100%)"],
  # ========================================================================== #
  # block identifier 'z'
  # ========================================================================== #
  genesis_z          => [227,1,"# genesis block identifier 'z'"],
  in_z               => [227,1,"# 227+(64*in_blocks) in block identifier 'z'"],
  coinbase_z         => [305,1,"# 305 coinbase block identifier 'z'"],
  out_z              => [363,1,"# 363  1  block identifier 'z'"]
};

################################################################################

1;



#################################################################################
# Read Blok data by names
#################################################################################
# read_tix ( filehandle, current-block-position, tick-name, [inblock-index-number] ) : raw_ledger_data
sub read_tix {
  my($fh,$cbp,$name,$num)=@_;
  if(defined $TIX->{$name}){
    $fh->seek( $cbp + $TIX->{$name}[0] + (defined $num ? $num * $TIX->{$name}[1] : 0 ) );
    return $fh->read($TIX->{$name}[1])
  }
}
#################################################################################



################################################################################
# genesis|inblock|coinbase|outblock Block Header
sub read_prev { my($fh,$bp)=@_; return hexdec(read_tix($fh,$bp,'prev')) }
sub read_next { my($fh,$bp)=@_; return hexdec(read_tix($fh,$bp,'next')) }
sub read_tid { my($fh,$bp)=@_; return read_tix($fh,$bp,'tid') }
sub read_cumhash { my($fh,$bp)=@_; return read_tix($fh,$bp,'cumhash') }
sub read_num { my($fh,$bp)=@_; return hexdec(read_tix($fh,$bp,'num')) }
sub read_version { my($fh,$bp)=@_; return read_tix($fh,$bp,'version') }
sub read_type { my($fh,$bp)=@_; return $RTRANSTYPES->{read_tix($fh,$bp,'type')} }
sub read_prevtid { my($fh,$bp)=@_; return read_tix($fh,$bp,'prevtid') }
sub read_fcctime { my($fh,$bp)=@_; return hexdec(read_tix($fh,$bp,'fcctime')) }
#################################################################################
# genesis|inblock|coinbase Block Content
sub read_outblocks { my($fh,$bp)=@_; return hexdec(read_tix($fh,$bp,'outblocks')) }
#################################################################################
# coinbase Block Content
sub read_coinbase_count { my($fh,$bp)=@_; return hexdec(read_tix($fh,$bp,'coinbase_count')) }
sub read_coinbase_signature { my($fh,$bp)=@_; return hexdec(read_tix($fh,$bp,'coinbase_signature')) }
# inblock|coinbase Block Content
sub read_in_blocks { my($fh,$bp)=@_; return hexdec(read_tix($fh,$bp,'in_blocks')) }
sub read_in_signature { my($fh,$bp)=@_; return hexdec(read_tix($fh,$bp,'in_signature')) }
sub read_in_block { my($fh,$bp,$num)=@_; return read_tix($fh,$bp,'in_block',$num) }
#################################################################################
# outblock Block Content
sub read_out_address { my($fh,$bp)=@_; return read_tix($fh,$bp,'out_address') }
sub read_out_amount { my($fh,$bp)=@_; return hexdec(read_tix($fh,$bp,'out_amount')) }
sub read_out_fee { my($fh,$bp)=@_; return hexdec(read_tix($fh,$bp,'out_fee')) }
#################################################################################


#################################################################################
# Node Saldo_Detail Request
sub saldo_detail {
  my ($wid) = @_;
  my $detail={};
  my $fh=gfio::open('ledger.fcc');
  my $wpl=walletposlist($wid);
  foreach my $obp (@$wpl) {
    if (read_out_address($fh,$obp) eq $wid) {
      my $amount=read_out_amount($fh,$obp);
      my $tid=read_tid($fh,$obp);
      my $ibp=scroll_back_to_inblock_pos($fh,$obp);
      my $fcctime=read_fcctime($fh,$ibp);
      my $fromaddr=read_fromaddress($fh,$ibp);
      $detail->{$tid} = { fcctime => $fcctime, amount => $amount, from => $fromaddr };
    }
  }
  $fh->close;
  return $detail
}
#################################################################################
# Seek Inblock from Outblock position
sub scroll_back_to_inblock_pos {
  my ($fh,$bp) = @_;
  my $type = read_type($fh,$bp);
  while ( $bp > 0 && $type eq 'out') {
    $bp -= read_prev($fh,$bp);
    $type = read_type($fh,$bp)
  }
  return $bp
}
#################################################################################
# Get Outblock From Address
sub read_fromaddress {
  my ($fh,$ibp)=@_;
  my $ibtp=read_type($fh,$ibp);
  if($ibtp eq 'in'){
    return read_out_address($fh,scroll_back_to_last_inblocks_outblock_pos($fh,$ibp));
  }
  elsif ($ibtp eq 'coinbase'){
    return $ibtp.':'.read_coinbase_count($fh, $ibp)
  }else{
    return $ibtp
  }
}
#################################################################################
# Seek last Inblocks Outblock from Inblock position
sub scroll_back_to_last_inblocks_outblock_pos {
  my ($fh,$bp) = @_;
  my $tid = read_in_block($fh,$bp,read_in_blocks($fh,$bp)-1);
  my $type = read_type($fh,$bp);
  while ( $bp > 0 && $type ne 'out' && read_tid($fh,$bp) ne $tid) {
    $bp -= read_prev($fh,$bp);
    $type = read_type($fh,$bp)
  }
  return $bp
}
#################################################################################



#################################################################################
# EOF FCC::tix (C) 2018 Chaosje, Domero
#################################################################################

