#!/usr/bin/perl

$SIG{__DIE__}=\&fatal;
$SIG{__WARN__}=\&fatal;

use Fcntl qw (:DEFAULT :flock);

my %OPENED=();
my $UPDATEDIR;

for my $d (@INC) { 
  if ($d =~ /site\/lib/) { $UPDATEDIR = $d }
}
if (!$UPDATEDIR) {
  $UPDATEDIR=$INC[0]
}

print <<EOT;

  FFFF  CCC   CCC
  F    C     C               INSTALLATION
  FF   C     C            
  F    C     C              (C) 2019 Domero
  F     CCC   CCC            mailto: factorialcoin@gmail.com
  
EOT

print "Installing Perl modules to $UPDATEDIR\n";

my $files=readfiles('modules','pm',1);
my $num=numfiles($files);
print "Transferring $num modules ... \n";
for my $f (1..$num) {
  my $file=getfile($files,$f);
  print "> Transferring $file->{fullname}\n";
  my $dir=$file->{dir};
  if ($dir =~ /FCC/) { $dir =~ s/^modules\///; $target="$dir/$file->{name}" }
  else { $target=$file->{name} }
  my $cont=content($file->{fullname});
  create("$UPDATEDIR/$target",$cont)
}

print "\nInstalling CPAN modules ... \n";
my $list = [
  'Compress::Raw::Zlib',
  'Crypt::Ed25519',
  'Digest::SHA',
  'JSON',
  'Time::HiRes',
  'URL::Encode',
  'MIME::QuotedPrint',
  'Compress::Bzip2',
  'Digest::MD5',
  'HTTP::Date',
  'Browser::Open',
  'IO::Socket::SSL'
];

for my $mod (@$list) {
  print "\n> CPAN Install $mod\n\n";
  system "cpan install $mod"
}

print "\n\nInstallation complete!\n\n";

exit;

sub fatal {
  error(@_)
}

sub error {
  my @msg=@_; push @msg,"";
  select(STDOUT); binmode(STDOUT); $|=1;
  print "\n "; print ("*"x32); print " FATAL ERROR "; print ("*"x33); print "\n";
  foreach my $line (@msg) {
    while (length($line)>0) {
      print " * ";
      if (length($line)>74) {
        my $disp=substr($line,0,71)."..."; print $disp; print " *\n";
        $line="...".substr($line,71)
      } else {
        print $line; print (" "x(74-length($line))); print " *\n";
        $line=""
      }
    }
  }    
  print " "; print ("*"x32); print " FATAL ERROR "; print ("*"x33); print "\n";
  print trace();
  exit 1
}

sub trace {
  my $i=1; my $out="";
  while (($i>0) && ($i<20)) {
    my ($package,$filename,$line,$subroutine,$hasargs,$wantarray,$evaltext,$is_require,$hints,$bitmask,$hinthash)=caller($i);
    if (!$package) { $i=0 }
    else {
      $out.="$package($filename): Line $line calling $subroutine\n";
      $i++
    }
  }
  return $out
}

sub fopen {
  my ($file,$mode) = @_;
  my $self = {}; bless $self;
  $self->{file}=$file;
  $self->{openmode}=$mode;
  $self->{read}=0; if (!defined $mode || ($mode =~ /r/i)) { $self->{read}=1 }
  $self->{fwrite}=0; if (defined $mode && $mode =~ /w/i) { $self->{fwrite}=1 }
  $self->{append}=0; if (defined $mode && $mode =~ /a/i) { $self->{append}=1 }
  if (!-e $file) {
    if (!$self->{fwrite} && !$self->{append}) {
      error("Open: File '$file' does not exist")
    } else {
      makepath($self);
      if ($self->{read}) {
        sysopen($self->{handle},$file,O_CREAT | O_RDWR | O_BINARY)
      } else {
        sysopen($self->{handle},$file,O_CREAT | O_WRONLY | O_BINARY)
      }
    }
  } else {
    if (!-f $file) {
      if (-d $file) {
        error("Open: Cannot overfwrite directory '$file' as a file")
      } elsif (-s $file) {
        error("Open: Cannot overfwrite symlink '$file' as a file")       
      } else {
        error("Open: Cannot overfwrite '$file' as a file")
      }
    }
  }
  if ($self->{append}) {
    if ($self->{read}) {
      sysopen($self->{handle},$file,O_APPEND | O_RDWR  | O_BINARY) || error("Open: Cannot open '$file' in mode 'ar': $!")
    } else {
      sysopen($self->{handle},$file,O_APPEND | O_WRONLY | O_BINARY) || error("Open: Cannot open '$file' in mode 'a': $!")
    }
  } elsif ($self->{fwrite}) {                                                                                      
    if ($self->{read}) {
      sysopen($self->{handle},$file,O_RDWR | O_BINARY) || error("Open: Cannot open '$file' in mode 'rw': $!")
    } else {
      sysopen($self->{handle},$file,O_WRONLY | O_BINARY) || error("Open: Cannot open '$file' in mode 'w': $!")
    }
  } else {
    sysopen($self->{handle},$file,O_RDONLY | O_BINARY) || error("Open: Cannot open '$file' in mode 'r': $!")
  }
  my @st=stat($self->{file}); $self->{size}=$st[7];
  if ($self->{append}) {
    $self->{position}=$self->{size}
  } else {
    $self->{position}=0
  }
  $self->{opened}=1;
  $self->{locked}=0;
  $OPENED{$file}=$self;
  return $self    
}

sub fcloseall {
  foreach my $file (keys %OPENED) { $OPENED{$file}->fclose }
}

sub makepath {
  my ($self) = @_;
  my @dir=split(/\//,$self->{file}); pop @dir;
  my $path="";
  foreach my $d (@dir) {
    $path.=$d;
    if ($path ne "" && $path ne '.' && $path ne '..' && !-e $path) {
      mkdir($path,0700)
    }
    $path.="/"
  }
}

sub filesize {
  my ($self) = @_; return $self->{size}
}
 
sub makedir {
  my ($newdir,$mode) = @_;
  if (!$newdir) { return }
  if(!$mode){ $mode=0700 }
  my @dir=split(/\//,$newdir);
  my $path="";
  foreach my $d (@dir) {
    $path.=$d;
    if ($path && ($path ne '.') && ($path ne '..') && (!-e $path)) {
      mkdir($path,$mode)
    }
    $path.="/"
  }
}

sub create {
  # WILL OVERfwrite!!!
  my ($filename,$content,$nonil,$mode) = @_;
  if (!$filename) { return }
  if ($nonil) {
    if(!defined $content) { return }
    if (ref($content) eq 'SCALAR') {
      if (length(${$content})==0) { return }
    } elsif (length($content)==0) { return }
  }
  if (-e $filename && -f $filename) {
    unlink($filename)
  } 
  my $fh=fopen($filename,'w');
  if ((defined $content) && (length($content))) {
    if (ref($content) eq 'SCALAR') { 
      fwrite($fh,$content)
    } 
    else { fwrite($fh,\$content) }
  }
  fclose($fh);
  if ($mode) { chmod $mode,$filename }
}

sub fseek {
  my ($self,$pos) = @_;
  if ($pos<0) {
    error("fseek: Trying to fseek before beginning of file '$self->{file}'","fseek=$pos EOF=$self->{size}")
  }  
  if ($pos>$self->{size}) {
    error("fseek: fseek beyond end of file '$self->{file}'","fseek=$pos EOF=$self->{size}")
  }
  sysseek($self->{handle},$pos,0); $self->{position}=$pos;
  return $self
}

sub newfile {
  my ($filename,$content,$noread) = @_;
  if (!$filename) { return }
  my $mode='rw'; if ($noread) { $mode='w' }
  my $fh=fopen($filename,$mode);
  if (ref($content) eq 'SCALAR') { fwrite($fh,$content) } else { fwrite($fh,\$content) }
  return $fh
}

sub append {
  my ($filename,$content) = @_;
  if (!-e $filename) {
    error("Append: File '$filename' does not exist")
  }
  my $fh=fopen($filename,'a'); 
  if (ref($content) eq 'SCALAR') { fwrite($fh,$content) } else { fwrite($fh,\$content) }
  fclose($fh)
}

sub content {
  my ($filename,$offset,$length) = @_;
  if (!$filename) {
    error("Content: No filename given")    
  }
  if (!-e $filename) {
    error("Content: File '$filename' does not exist")
  }
  if (!-f $filename) {
    error("Content: '$filename' is not a plain file")
  }
  my $fh=fopen($filename,'r');
  if (!defined $offset) { $offset=0 }
  if (!defined $length) { $length=$fh->{size} }
  if ($offset>$fh->{size}) {
    error("Content: Read beyond boundries of '$filename', offset=$offset, size=$fh->{size}")
  }
  if ($offset+$length>$fh->{size}) {
    error("Content: Read beyond boundries of '$filename', offset=$offset, reading $length bytes, size=$fh->{size}")
  }
  fseek($fh,$offset);
  my $txt=readptr($fh,$length); fclose($fh);
  return ${$txt}
}

sub changeowner {
  if ($^O =~ /win/i) { return }
  my ($filename,$user,$group) = @_;
  my ($login,$pass,$uid,$gid) = getpwnam($user);
  my ($glogin,$gpass,$guid,$ggid) = getpwnam($group);
  if ($uid && $ggid && $filename) {
    chown $uid, $ggid, $filename
  }
}

sub copy {
  my ($src,$des,$nooverfwrite)=@_;
  if (!$src || !$des) { return }
  if (!-e $src) {
    error("Copy: Source file '$src' does not exist")
  }
  if (!-f $src) {
    error("Copy: Source '$src' is not a file!")
  }
  if (!$nooverfwrite || !-e $des) {
    if (-e $des) { unlink $des }
    my $s=fopen($src,'r'); my $d=fopen($des,'w');
    my $eof=0; my $b=1<<20; my $p=0; my $l=$s->{size};
    while(!$eof){
      if($p+$b>$l){ $b=$l-$p }
      fwrite($d,readptr($s,$b));
      $p+=$b; if($p>=$l){ $eof=1 }
    }
    fclose($s); fclose($d);
  }
}

sub fclose {
  my ($self) = @_; 
  if ($self->{opened}) {
    my $oldh = select $self->{handle}; $| = 1; select($oldh); # flush
    while ($self->{locked}) { unlock($self) }
    close($self->{handle}); $self->{opened}=0;
    delete $OPENED{$self->{file}}
  }
}

sub ftell {
  my ($self) = @_; return $self->{position}
}

sub read {
  my ($self,$len,$stopatend) = @_;
  if (!$len) { return "" }
  if (!$self->{opened}) { error("Read: File '$self->{file}' is fclosed") }
  if (!$self->{read}) { error("Read: File '$self->{file}' is read-protected") }
  if ($self->{position}+$len>$self->{size}) {
    if ($self->{position}>$self->{size}) {
      error("Read: Trying to read beyond the end of file '$self->{file}', position=$self->{position} len=$len size=$self->{size}")
    } elsif ($stopatend) {
      my $mlen=$self->{size}-$self->{position};
      if ($len>$mlen) { $len=$mlen }
    } else {
      error("Read: Trying to read beyond the end of file '$self->{file}', position=$self->{position} len=$len size=$self->{size}")
    }
  }
  sysseek($self->{handle},$self->{position},0) || error("Read: Error fseeking in file '$self->{file}' pos=$self->{position}: $!");
  my $data;
  if ($len<0) {
    error("Read: Neagtive length '$len' reading on position '$self->{position}' in file '$self->{file}', size=$self->{size}")
  }
  sysread($self->{handle},$data,$len) || error("Read: Error reading from file '$self->{file}', len=$len: $!");
  $self->{position}+=$len;
  return $data;
}

sub readptr {
  my ($self,$len,$errormode) = @_; 
  if (!$len) { my $dat=""; return \$dat }
  if (!$self->{opened}) { error("ReadPtr: File '$self->{file}' is fclosed") }
  if (!$self->{read}) { error("ReadPtr: File '$self->{file}' is read-protected") }
  if ($self->{position}>$self->{size}) {
    $self->{position}=$self->{size}
  }
  if ($self->{position}+$len>$self->{size}) {
    if ($errormode) {
      error("ReadPtr: Trying to read beyong boundries of file '$self->{file}', position=$self->{position} len=$len size=$self->{size}")
    } else {
      $len=$self->{size}-$self->{position};
    }  
  }
  sysseek($self->{handle},$self->{position},0) || error("ReadPtr: Error fseeking in file '$self->{file}' pos=$self->{position}: $!");
  my $data;
  sysread($self->{handle},$data,$len) || error("ReadPtr: Error reading from file '$self->{file}', len=$len: $!");
  $self->{position}+=$len;
  return \$data;
}

sub readlines {
  my ($filename) = @_;
  if (!$filename) {
    error("Content: No filename given")    
  }
  if (!-e $filename) {
    error("Content: File '$filename' does not exist")
  }
  if (!-f $filename) {
    error("Content: '$filename' is not a plain file")
  }
  my $size=0; my $txt;
  my $fh=fopen($filename,'r'); $size=$fh->{size}; $txt=$fh->read($size); $fh->fclose;
  my $lines=[]; my $i=0; my $curline="";
  while ($i<$size) {
    my $c=substr($txt,$i,1); my $cc=ord($c);
    if ($cc!=13) {
      if ($cc==10) {
        push @{$lines},$curline; $curline=""
      } else {
        $curline.=$c
      }
    }
    $i++
  }
  return $lines
}

sub fwrite {
  my ($self,$data,$nonil) = @_;
  if (ref($data) eq 'SCALAR') { $data=${$data} }
  if (!defined $data) { 
    if ($nonil) { error("fwrite: Trying to fwrite empty data, while prohibited") }
    return $self
  }
  if (!$self->{opened}) { error("fwrite: File '$self->{file}' is fclosed") }
  if (!$self->{fwrite} && !$self->{append}) { error("fwrite: File '$self->{file}' is fwrite-protected") }
  sysseek($self->{handle},$self->{position},0) || error("fwrite: Error fseeking in file '$self->{file}' pos=$self->{position}: $!");
  syswrite($self->{handle},$data) || error("fwrite: Error writing to file '$self->{file}', len=".length($data).": $!");
  $self->{position}+=length($data);
  if ($self->{position}>$self->{size}) { $self->{size}=$self->{position} }
  return $self
}


sub ftruncate {
  my ($self,$length) = @_;
  if ($self->{size}<=$length) { return }
  truncate($self->{handle},$length);
  if ($self->{position}>$length) { $self->{position}=$length }
  $self->{size}=$length;
  return $self
}

sub lock {
  my ($self) = @_;
  if (!$self->{locked}) {
    flock($self->{handle},LOCK_EX)
  }  
  $self->{locked}++;
  return $self
}

sub unlock {
  my ($self) = @_;
  if ($self->{locked}) {
    $self->{locked}--;
    if (!$self->{locked}) {
      flock($self->{handle},LOCK_UN)
    }
  } else {
    error("Unlock: File '$self->{file}' was not locked!")
  }
  return $self
}

sub locked {
  my ($self) = @_;
  return $self->{locked}
}

sub insert {
  my ($self,$content,$append) = @_;
  if ($append) { fseek($self,$self->{size}) }
  my $start=$self->{position};
  if (!$self->{fwrite} && !$self->{append}) { error("Insert: File '$self->{file}' is fwrite-protected") }
  my $movelen=$self->{size}-$start;
  my $dat=readptr($self,$movelen);
  fseek($self,$start); fwrite($self,$content); 
  my $pos=ftell($self); fwrite($self,$dat); fseek($self,$pos)
}

sub appenddata {
  my ($self,$content) = @_;
  if (!$self->{fwrite} && !$self->{append}) { error("Appenddata: File '$self->{file}' is fwrite-protected") }
  fseek($self,$self->{size});
  fwrite($self,$content)
}

sub extract {
  my ($self,$len) = @_;
  my $start=$self->{position};
  my $pos=$self->{position}+$len;
  my $dat;
  if ($pos>$self->{size}) {
    $dat=readptr($self,$self->{size}-$start);
    ftruncate($self,$start)
  } else {
    fseek($self,$pos); $dat=readptr($self,$self->{size}-$pos);
    fseek($self,$start); fwrite($self,$dat); ftruncate($self,$self->{size}-$pos); fseek($self,$start)
  }
  return ${$dat}
}

sub verbosefile {
  my ($self,$txt) = @_;
  print "\rReading: ";
  if (length($txt)>70) {
    print "...".substr($txt,length($txt)-67)
  } else {
    print $txt; print " "x(70-length($txt))
  }
}

sub doreadfiles {
  my ($self,$dir,$verbose,$num) = @_;
  my $fl; my $handle;
  opendir($handle,$dir) or error("Readfiles: Error opening directory '$dir': $!");
  my $slash=(substr($dir,length($dir)-1,1) eq '/');
  do {
    my $ff;
    $fl=readdir($handle);
    if ($fl && ($fl ne ".") && ($fl ne '..')) {
      my @ss=split(/\//,$fl); my $fname=pop @ss;
      if ((lc($fname) ne 'system volume information') && (lc($fname) ne 'recycler')) {
        my @ps=split(/\./,$fname); my $fext=pop @ps; my $fsname;
        if ($fname =~ /\./) { $fsname=join(".",@ps) } else { $fsname=$fext; $fext="" }
        if ($slash || !$dir) { $ff=$dir.$fname } else { $ff=$dir."/".$fname }
        if ((!-l $ff) && (-r $ff)) {
          if (-d $ff) {
            if ($self->{recursive}) {
              if ($verbose) { verbosefile($self,"[$ff]") }
              doreadfiles($self,$ff,$verbose,$num)
            }
          } elsif ($self->{allext} || $self->{extlist}{lc($fext)}) {
            if ($verbose) { verbosefile($self,"${$num}. $fname") }
            ${$num}++;
            my @data=($fsname,$fext,$fname,$dir,$ff,'file');
            push @{$self->{list}},\@data;
          }
        }
      }
    }
  } until (!$fl);
  closedir($handle)
}

sub readfiles {
# INPUT    dir,"ext,ext,..",subdirs also (recursively), verbose (if 1 prints info to the <STDOUT>)
# Usage: $files->{list}[num][ 0=name, 1=extension, 2=name+ext, 3=directory, 4=dir+name+ext ]
  my ($dir,$extlist,$subdirs,$verbose) = @_;
  $dir =~ s/\\/\//g;
  my $self={}; bless($self);
  $self->{dir}=$dir; $self->{exist}=1;
  $self->{list}=[]; $self->{recursive}=$subdirs;
  if (defined($extlist)) {
    $extlist =~ s/ //g; $self->{extlist}={};
    foreach my $ext (split(/\,/,$extlist)) {
      $self->{extlist}{lc($ext)}=1
    }
  }  
  if (!-e $dir) {
    $self->{exist}=0; return $self
  }
  $self->{allext}=(!defined($extlist) || ($extlist eq '*') || !$extlist);
  my $num=1;
  doreadfiles($self,$dir,$verbose,\$num);
  if ($verbose) { print "\r"; print " "x79; print "\r" }
  return $self
}

sub numfiles {
  my ($self) = @_;
  return 0+@{$self->{list}}
}

sub getfile {
  my ($self,$num) = @_;
  if (!$num) { $num=0 }
  if (($num<1) || ($num>$self->numfiles)) {
    error("GetFile: File '$num' is invalid (must be between 1 and ".numfiles($self).", reading '".$self->{dir}."')")
  }
  my $fi=$self->{list}[$num-1];
  my @stat=stat($fi->[4]) || (0)x11;
  my $info={
    barename => $fi->[0],
    ext => $fi->[1],
    name => $fi->[2],
    dir => $fi->[3],
    fullname => $fi->[4],
    level => $fi->[5],
    mode => $stat[2],
    size => $stat[7],
    atime => $stat[8],
    mtime => $stat[9],
    ctime => $stat[10]
  };
  $info->{isdir}=0; if ($fi->[5] =~ /[0-9]/) { $info->{isdir}=1 }
  return $info
}

